
/* -*-C-*-
 ##############################################################################
 #
 # File:        e1430/example/example.c
 # RCS:         "@(#)$Revision: 1.1 $"
 # Description: Example program for HP E1430 
 # Language:    C
 #
 # (C) Copyright 1994, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This program demonstrates how to acquire data from the HP E1430 module
 # over the VXI bus.  The program uses functions in the 
 # "E1430 C Interface Library", and so must be linked with this library.  
 #
 # The program can run on several computer platforms, including:
 # 	An embedded HP-UX VXI computer like the V/382
 # 	An external HP-UX computer with MXI connection to VXI
 # 	An embedded DOS VXI computer like the RadiSys EPC-7
 # 	A Digital Signal Processor VXI module such as the E1485
 #
 # Revisions:
 #
 ##############################################################################
*/

#include "machType.h"

#ifdef E1485_SOURCE
#  include "spil_d.h"
#else
#  include <stdio.h>
#  include <stdlib.h>
#endif

#include "e1430.h"

#define E1430_LA     129        /* VXI logical address of HP E1430 module */
#define BLOCK_SIZE   1024       /* Measurement blocksize      */

FLOATSIZ32 dataBuffer[BLOCK_SIZE];
ULONGSIZ32 dataBytes = BLOCK_SIZE * sizeof( FLOATSIZ32 );
SHORTSIZ16 groupID;            /* ID used by HP E1430 libraries           */

/****************************************************************************
 *
 > setUpInputs()
 *
 * $Description: $
 *    This routine does all the necessary setup on the E1430 input module.
 *
 * $Return:     (void) $
 *
 ****************************************************************************/

void setUpInputs( void )
{
    SHORTSIZ16 error, la = E1430_LA;

    /* Initialize the E1430 libraries, and reset the module */

    /* Enable (1) or disable (0) e1430 debug printfs */
    e1430_debug_level(0); 

    error = e1430_init_io_driver();
    if( error )
    {
        (void)printf( "e1430_init_io_driver() error: %d, %s\n",
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    groupID = e1430_create_module_group( 1, &la );
    if( groupID <= 0 )
    {
       (void)printf( "Can't create module group with module at la = %d, %s\n",
                       la, e1430_get_error_string() );
        exit( -1 );
    }

    error = e1430_reset_module( groupID );
    if( error )
    {
        (void)printf( "e1430_reset_module() error: %d, %s\n",
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    /* Set the range and other analog front end parameters */

    error = e1430_set_analog_input( groupID,
                                    2.0, /* range in Vp */
                                    E1430_COUPLING_DC,
                                    E1430_ANTIALIAS_ON,
                                    E1430_INPUT_HI_CONN,
                                    E1430_INPUT_LO_FLOAT );
    if( error )
    {
        (void)printf( "e1430_set_analog_input() error: %d, %s\n",
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    /* Set the block size and other data format parameters           */

    error = e1430_set_data_format( groupID,
                                   E1430_DATA_TYPE_REAL,
                                   E1430_DATA_SIZE_32,
                                   E1430_BLOCK_MODE,
                                   BLOCK_SIZE, 
                                   E1430_APPEND_STATUS_OFF );
    if( error )
    {
        (void)printf( "e1430_set_data_format() error: %d, %s\n",
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    /* Set the frequency span by enabling the decimation filter.  */

    error = e1430_set_decimation_filter( groupID, 
				1,   /* filter level.  1= 2MHz Bandwidth */
				E1430_DECIMATION_ON,
				E1430_ONEPASS,
				E1430_PASS_TAG_32 );

    if( error )
    {
        (void)printf( "e1430_set_decimation_filter() error: %d, %s\n", 
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    /* Autozero the inputs to remove DC offset */

    error = e1430_auto_zero( groupID );
    if( error )
    {
        (void)printf( "e1430_auto_zero() error: %d, %s\n",
                       error, e1430_get_error_string() );
        exit( -1 );
    }

    /* Note:  After a reset or autozero, the E1430 requires 20 seconds 
     * to allow the ADC correction values to settle to specified 
     * accuracy.   After code development, a delay should be inserted here.
     */ 
    return;
}


/*************************** Main program block ****************************/

int main( void )
{
    SHORTSIZ16 error, adcOvld, adcErr;
    LONGSIZ32  actualCnt, i;

    (void)printf("E1430 example program.\n" );
    (void)printf("Setting up ...\n" );

    setUpInputs( );
    
    (void)e1430_display_module_state( groupID );

    (void)printf("Begin to acquire data ...\n" );
    error = e1430_arm_module( groupID );
    if( error )
    {
	(void)printf( "e1430_arm_module() error: %d, %s\n",
		      error, e1430_get_error_string() );
        exit( -1 );
    }

    (void)printf("Wait for data, then transfer data ...\n" );
    error = e1430_read_float32_data( E1430_LA, dataBuffer, dataBytes,
				     &adcOvld, &adcErr,    &actualCnt );
    if( error )
    {
	(void)printf( "e1430_read_float32_data() error: %d, %s\n",
		      error, e1430_get_error_string() );
        exit( -1 );
    }

    (void)printf("Time record samples (in Volts): \n");
    for (i = 0; i < 40; i += 4)
    {
        (void)printf("%3d: %10.6f  %10.6f  %10.6f  %10.6f \n", (int)i,
		(double)dataBuffer[i+0], (double)dataBuffer[i+1], 
		(double)dataBuffer[i+2], (double)dataBuffer[i+3] );
    }

    /* delete E1430 module groups to free memory */
    error = e1430_delete_all_module_groups();
    
    return( 0 );
}
